# 機能設計書 8-状態管理（State Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおける状態管理（State Operations）機能の設計を記述する。Variable・Assign・ScatterUpdateなどのステートフル操作を提供する。

### 本機能の処理概要

状態管理機能は、TensorFlowにおけるレガシー変数（Variable/VariableV2）の作成・代入・更新・初期化確認などのステートフル操作を提供する。リソース変数（ResourceVariable）が導入される以前の変数管理方式であり、Ref型テンソルを通じた変数操作を実現する。

**業務上の目的・背景**：TensorFlowの初期バージョン（v1系）では、モデルパラメータはVariable/VariableV2 opで管理されていた。後方互換性の維持のために、これらのレガシー操作が引き続きサポートされている。Assign, AssignAdd, AssignSub, ScatterUpdate等の操作により、グラフ実行モードでの変数更新を実現する。

**機能の利用シーン**：(1) TF1.x互換コードでの変数管理、(2) レガシーモデルの変数更新、(3) 一時変数（TemporaryVariable）の使用、(4) ScatterUpdateによるスパース更新。

**主要な処理内容**：
1. Variable / VariableV2: 変数の作成
2. Assign: 変数への値代入
3. AssignAdd: 変数への加算代入
4. AssignSub: 変数からの減算代入
5. ScatterUpdate / ScatterAdd / ScatterSub / ScatterMul / ScatterDiv / ScatterMin / ScatterMax: スキャッター更新
6. ScatterNdUpdate / ScatterNdAdd / ScatterNdSub: N次元スキャッター更新
7. IsVariableInitialized: 変数初期化状態の確認
8. TemporaryVariable / DestroyTemporaryVariable: 一時変数の作成・破棄
9. CountUpTo: カウンタ変数のインクリメント

**関連システム・外部連携**：ResourceVariable ops（機能No.7）が後継機能。レガシーコードとの互換性のためにサポートが継続されている。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

CRUD操作（レガシー変数のCreate/Read/Update/Delete）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ref | Ref(T) | Yes | 変数参照テンソル | 有効なRef型テンソル |
| value | tf.Tensor | Assign時 | 代入値 | 変数と同じdtype |
| shape | tf.TensorShape | Variable時 | 変数の形状 | 有効な形状 |
| dtype | tf.DType | Yes | データ型 | サポートされるデータ型 |
| validate_shape | bool | No | 代入時の形状検証 | デフォルトtrue |
| use_locking | bool | No | ロック使用フラグ | デフォルトtrue |
| indices | tf.Tensor(int32/int64) | Scatter時 | 更新対象インデックス | 有効なインデックス |
| updates | tf.Tensor | Scatter時 | 更新値 | indicesと整合する形状 |

### 入力データソース

Python API呼び出し時の引数として入力される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ref | Ref(T) | Variable/Assign: 変数参照 |
| value | tf.Tensor | DestroyTemporaryVariable: 変数の最終値 |
| is_initialized | tf.Tensor(bool) | IsVariableInitialized: 初期化状態 |

### 出力先

呼び出し元のPythonコードに返却される。

## 処理フロー

### 処理シーケンス

```
1. Variable/VariableV2 op実行
   └─ shape/dtype属性に基づいて変数メモリ確保
2. Assign op実行（初期値代入）
   └─ validate_shape=trueの場合、形状一致を検証
3. 学習ループ内での操作
   └─ Assign/AssignAdd/AssignSub/ScatterUpdate
4. 変数状態の確認
   └─ IsVariableInitialized
```

### フローチャート

```mermaid
flowchart TD
    A[VariableV2 op] --> B[メモリ確保]
    B --> C[Assign:初期値代入]
    C --> D{操作種別}
    D -->|代入| E[Assign]
    D -->|加算代入| F[AssignAdd]
    D -->|スキャッター| G[ScatterUpdate系]
    D -->|初期化確認| H[IsVariableInitialized]
    E --> I[Ref(T)返却]
    F --> I
    G --> I
    H --> J[bool返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 形状検証 | validate_shape=trueの場合、Assignの値形状が変数形状と一致 | Assign操作時 |
| BR-02 | 未初期化許容 | SetAllowsUninitializedInputにより未初期化変数への代入を許容 | Assign/IsVariableInitialized |
| BR-03 | ステートフル | Variable/VariableV2はSetIsStatefulで登録 | 常時 |
| BR-04 | ロック制御 | use_locking=trueでスレッドセーフな更新 | 更新操作時 |
| BR-05 | レガシー形状推論 | Variableのscalar shapeは「unknown shape」として扱う | Variable op時 |

### 計算ロジック

- Assign: variable = value
- AssignAdd: variable = variable + value
- AssignSub: variable = variable - value
- ScatterUpdate: variable[indices] = updates
- ScatterAdd: variable[indices] += updates
- CountUpTo: variable += 1（limit到達時エラー）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | 形状不一致 | validate_shape=trueで形状が異なる | 形状を一致させるか、validate_shape=falseにする |
| OutOfRangeError | 範囲超過 | CountUpToでlimitに到達 | limitを増やすか、リセットする |
| FailedPreconditionError | 未初期化 | 未初期化変数の読み取り | Assignで初期化してからアクセスする |

### リトライ仕様

該当なし。

## トランザクション仕様

use_locking=trueの場合、個別の変数更新はミューテックスによりアトミックに実行される。

## パフォーマンス要件

- ScatterUpdate系操作はスパースインデックスに対して効率的に実行すること
- use_locking=falseの場合、ロックオーバーヘッドなしで実行すること

## セキュリティ考慮事項

- 特段のセキュリティリスクなし

## 備考

- State Opsはレガシー機能であり、新規コードではResourceVariable Ops（機能No.7）の使用が推奨される
- Variable opのscalar shape属性は「unknown shape」と区別できないレガシー動作がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Ref型テンソルと通常テンソルの違い、および変数のメモリモデルを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | state_ops.cc | `tensorflow/core/ops/state_ops.cc` | Variable/VariableV2のAttr定義とRef型出力 |

**読解のコツ**: State Opsの出力は `Ref(T)` 型であり、変数メモリへの参照を表す。これはResourceVariable opsの `resource` ハンドルとは異なるレガシーな仕組みである。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | state_ops.cc | `tensorflow/core/ops/state_ops.cc` | REGISTER_OPによるop定義 |

**主要処理フロー**:
- **24-31行目**: `VariableV2` op登録 - shape/dtype属性、ExplicitShape形状推論
- **33-55行目**: `Variable` op登録 - scalar shapeのレガシー動作
- **57-62行目**: `IsVariableInitialized` op登録 - bool出力のスカラー
- **64-70行目**: `TemporaryVariable` op登録 - 一時変数
- **72-77行目**: `DestroyTemporaryVariable` op登録
- **79-96行目**: `Assign` op登録 - validate_shape/use_locking属性、AllowsUninitializedInput
- **98-100行目以降**: `AssignAdd`, `AssignSub` op登録

### プログラム呼び出し階層図

```
tf.compat.v1.Variable(initial_value, ...)
    |
    +-- VariableV2 op
    |       +-- ExplicitShape (形状推論)
    |       +-- メモリ確保
    |
    +-- Assign(ref, initial_value)
            +-- MergeBothInputsShapeFn (validate_shape=true時)

variable.assign_add(delta)
    |
    +-- AssignAdd(ref, delta)
            +-- UnchangedShape (形状推論)
```

### データフロー図

```
[入力]                    [処理]                        [出力]

initial_value      -->  VariableV2                -->  Ref(T)
(テンソル)               (メモリ確保)
                            |
                    Assign(ref, value)           -->  Ref(T)
                    (値代入)
                            |
                    AssignAdd(ref, delta)         -->  Ref(T)
                    (加算代入)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| state_ops.cc | `tensorflow/core/ops/state_ops.cc` | ソース | C++ op定義 |
| gen_state_ops.py | `tensorflow/python/ops/gen_state_ops.py` | 自動生成 | Python バインディング |
| variables.py | `tensorflow/python/ops/variables.py` | ソース | Python Variable クラス |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 形状推論関数 |
